/************************************************************************
 * NAME:	floppy
 *
 * DESCR:	This is the standard floppy structure that is used for
 *		all formats.
 ************************************************************************/

/* "encoding" is use to indicate what the encoding for a particular sector is.	*/
/*    This can be FM or MFM, or something else.  The encoding is used in two	*/
/*    places:  on the sectors themselves and in the overall floppy structure.	*/
/*    The former is use to generate the sectors out from the SVD in the right	*/
/*    format, and the latter is used to generate other things such as trailers.	*/
/*    Note that in whacky copy protection schemes (like for the TRS-80) there	*/
/*    are mixed sectors so there needs to be a way to encode one sector in one	*/
/*    fashion and another sector in another fashion...even though the overall	*/
/*    format is one or the other.						*/
/*    "blank" encoding indicates that the given sector should be generated	*/
/*    according to the overall floppy format as a blank sector.  Normally, this	*/
/*    means generating blank data roughly akin to the trailer for the overall	*/
/*    floppy format.  This only comes up when there are an uneven number of	*/
/*    sectors for a particular floppy and the blanks are used to fill-in the	*/
/*    rest of the data structures when sending to the SVD.			*/

/* NOTE - that the definition of "enum" is such that the enumerations are	*/
/*   numbered from 0 to n in the order of definition.  Therefore, the last one	*/
/*   "UNKNOWN_ENCODING" is also the count of the number of enumerations.	*/

typedef enum {
    WD_FM,			/* Western Digital 1791 - single density	*/
    WD_MFM,			/* Western Digital 1793 - double density	*/
    H17_HSFM,			/* Heathkit H8/H17 - hard sector single density	*/
    RX02,			/* DEC RX02 - need more details here!		*/
    BLANK_SECTOR,		/* a simple blank sector			*/
    AGCR6x2,			/* apple GCR 6x2				*/
    AGCR5x3,			/* apple GCR 5x3				*/
    RNIB,			/* apple "raw" format - 26 sectors of nib data	*/
    UNKNOWN_ENCODING		/* always keep last - just a place holder	*/
} encoding;

struct sector {
    int			id;		/* this is really the track #		*/
    int			side;
    int			sector;
    int			sizecode;       /* doubles as the "volume" for h17/apple*/
    int			size;
    int			mark;		/* mark for sector 0xf8 - 0xfb		*/
                                        /* doubles as the mark for data/header on h17	*/

    unsigned short	headCRC;
    unsigned short	dataCRC;

    encoding		encoding;	/* this sector's encoding		*/

    unsigned char	data[256];	/* should be enuf fer now		*/

    /*************** APPLE 2 ****************************************************/
    /* apple 2 specific data - could probably overlay this with other data	*/
    /* but this is easy, and storage really isn't a big concern			*/

    int			hdrprolog[3];	/* header prolog bytes			*/
    int			hdrepilog[3];	/* header epilog bytes			*/
    int			volume;
    int			hdrchecksum;   
    int			datachecksum;	/* gcr-encoded checksum			*/
    unsigned char	preload;	/* preload data for sector		*/
    int			dataprolog[3];	/* data prolog bytes			*/
    int			dataepilog[3];	/* data epilog bytes			*/
};

struct track {
    int			sectors;	/* count of sectors on this track	*/
    unsigned char	sectormap[256];	/* a logical map of sectors (see below)	*/
    struct sector	*sector;	/* a track just has sectors		*/
};

/**** NOTE:  sectormap[] - this map is used to quickly access the sectors of a	*/
/****			   track.  So sectormap[0] contains the physical sector	*/
/****			   number where the sector with sector==0 can be found.	*/


struct floppy {
    int		write_protect;		/* 0xff is write-protected, 0x00 allows	*/
    int		tracks;			/* number of tracks on disk		*/
    int		sides;			/* either 1 or 2 sides			*/
    encoding	encoding;		/* specifies this disk's encoding	*/
    int		volume;			/* used for h17 format only		*/
    int		sectors;		/* number of sectors			*/
    int		secsize;		/* size of sectors in 128-byte chunks:	*/
                                        /* 0 = 128, 1 = 256, 3 = 512...		*/
                                        /* UNUSED FOR NOW			*/

    struct track	*side[2];	/* sectors for up to 2 sides		*/
};

/* sector marks	for SD */

#define FLOPPY_SEC_DELETED	0xf8
#define FLOPPY_SEC_UNKNOWN	0xf9
#define FLOPPY_SEC_SPECIAL	0xfa
#define FLOPPY_SEC_NORMAL	0xfb

#define SectorMap(f,s,t,sect)	   f->side[s][t].sectormap[sect]
#define LogicalSector(f,s,t,sect)  f->side[s][t].sector[SectorMap(f,s,t,sect)]
#define PhysicalSector(f,s,t,sect) f->side[s][t].sector[sect]

/* FILE EXTENSION REGISTERING							*/
/*										*/
/*	File extensions are registered according to the "strength" of the	*/
/*	guessor for that format and the specificity of the file extension. That	*/
/*	is, the file extension can be very specific and/or the guessor for the	*/
/*	format is very good (low number of false positives and negatives).  The	*/
/*	classification is used to order the guessors for a given extension.	*/
/*
/*	EXT_VERY_SPECIFIC - the file extension can be guessed with 100%		*/
/*				accuracy - if the guessor says "yes" then you	*/
/*				are SURE it is the right format.		*/
/*	EXT_SOMEWHAT_SPECIFIC - the file extension is probably used ONLY for	*/
/*				the registered format, or the guessor is pretty	*/
/*				good, or both.					*/
/*	EXT_NOT_SPECIFIC - the file extension MAY be this format, and the	*/
/*				guessor won't help too much.  These types of	*/
/*				formats are tried last.				*/

#define EXT_VERY_SPECIFIC	1
#define EXT_SOMEWHAT_SPECIFIC	2
#define EXT_NOT_SPECIFIC	3

/* sectormap associated defines	*/
/* Undefined entries in the sectormap are marked with 255 - this restricts the	*/
/* sector numbers to those from 0 to 254.					*/

#define SECTORMAP_UNDEFINED	255

extern int   floppy_fileext_register(char *, int, char *);

extern int   floppy_format_register(char *, 
				    char *,
				    int (*)(int), 
				    int (*)(int,struct floppy *),
				    int (*)(int,struct floppy *),
				    int (*)(int,struct floppy *,int));

extern char *floppy_format_guesser(int, char*);
extern int   floppy_format_reader(char *, int, struct floppy *);
extern int   floppy_format_dumper(char *, int, struct floppy *);
extern int   floppy_format_report(char *, int, struct floppy *, int);
extern char *floppy_format_list_string(char *, int, int);
extern char *floppy_format_list_select(char *, int);
extern char *floppy_format_add_params(char *);
extern void  floppy_copy_track(struct floppy *, int, int);

extern int   floppy_crc_check(struct floppy *, int, int, char *);
extern int   floppy_crc_set(struct floppy *);
extern char *floppy_file_extension(char *);


